module net.BurtonRadons.dedit.projectView;

import net.BurtonRadons.dig.main;
import net.BurtonRadons.dedit.main;
import net.BurtonRadons.dedit.document;

class ProjectView : TreeBox
{
    bit deferPaint = false;
    Folder root;
    
    this (Control parent)
    {
        super (parent);
        multipleSelection (true);
    }
    
    static void menu (Folder base)
    {
        MenuCommand menu = new MenuCommand ();
        
        menu.add ("New File", "FileNew");
        menu.add ("Open File", "FileOpen");
        menu.separator ();
        menu.add ("New Project", "ProjectNew");
        menu.add ("Open Project", "ProjectOpen");
        menu.add ("Save Project", "ProjectSave");
        menu.add ("Statistics", "ProjectStats");
        menu.separator ();
        menu.add ("New Folder", "ProjectNewFolder");
        menu.add ("Rename Folder", "ProjectRenameFolder");
        menu.add ("Delete Folder", "ProjectDeleteFolder");
        
        global.view.currentFolder (base);
        menu.run (global.view);
        global.view.currentFolder (null);
    }
    
    static void menu (Document base)
    {
        global.view.switchDocument (base);
        
        MenuCommand menu = new MenuCommand ();
        menu.add ("New", "FileNew");
        menu.add ("Open", "FileOpen");
        menu.add ("Close", "FileClose");
        menu.add ("Save", "FileSave");
        menu.add ("Rename", "FileRename");
        menu.separator ();
        menu.add ("New Project", "ProjectNew");
        menu.add ("Open Project", "ProjectOpen");
        menu.add ("Save Project", "ProjectSave");
        menu.add ("Statistics", "ProjectStats");
        menu.separator ();
        menu.add ("New Folder", "ProjectNewFolder");
        menu.run (global.view);
    }
    
    class Folder : TreeBox.TextFolder
    {
        private char [] privateName; /**< Name of the folder. */
        
        this (char [] name)
        {
            privateName = name.dup;
            onRButtonUp.add (&doRButtonUp);
            super (null);
        }
        
        override char [] columnText (Column column)
        {
            return name ();
        }
        
        /** Return the name of the folder. */
        char [] name ()
        {
            return privateName;
        }
        
        /** Assign the name of the folder, copying. */
        void name (char [] value)
        {
            privateName = value.dup;
            sortChange ();
        }
        
        void doRButtonUp ()
        {
            menu (this);
        }
        
        /** Return the number of child folders to this folder. */
        int folderCount ()
        {
            int count;
         
            folderIterate (delegate void (Folder row) { count ++; });   
            return count;
        }
        
        /** Call func on each folder in this folder. */
        void folderIterate (void delegate (Folder row) func)
        {
            for (int c; c < rowCount (); c ++)
                if ((Folder) row (c))
                    func ((Folder) row (c));
        }
        
        /** Return the number of child documents to this folder. */
        int documentCount ()
        {
            int count;
            
            documentIterate (delegate void (DocumentRow row) { count ++; });
            return count;
        }
        
        /** Call func on each document in this folder. */
        void documentIterate (void delegate (DocumentRow row) func)
        {
            for (int c; c < rowCount (); c ++)
                if ((DocumentRow) row (c))
                    func ((DocumentRow) row (c));
        }
    }
    
    class DocumentRow : TreeBox.TextRow
    {
        Document base;
        
        this (Document base)
        {
            this.base = base;
            super (null);
            
            onFocus.add (&doFocus);
            onRButtonUp.add (&doRButtonUp);
        }
        
        override char [] columnText (Column column)
        {
            return base.name ();
        }
        
        void doRButtonUp ()
        {
            menu (base);
        }
        
        void doFocus ()
        {
            global.view.switchDocument (base);
        }
        
        override void displayColumn (Column column, int x, int y, int width)
        {
            if (base.modified)
                control.textColor (AColor (255, 128, 128));
            
            if (base === global.view.document)
                control.textPrintEllipses (x + 1, y, width, 0, columnText (column));
            
            control.textPrintEllipses (x, y, width, 0, columnText (column));
        }
    }
    
    void doRootMenu ()
    {
        menu (root);
    }
    
    override void paint ()
    {
        if (deferPaint)
            return;
        super.paint ();
    }
    
    /** Paint a single document. */
    void paintDocument (Document document)
    {
        if (deferPaint)
            return;
        DocumentRow row = findDocument (document);
        
        if (row)
            paintRow (row);
    }
    
    /** Empty the tree and create the root folder. */
    void restart ()
    {
        empty ();
        root = new Folder ("Files");
        add (root);
        root.open (true);
        deferPaint = false;
        paint ();
    }
    
    DocumentRow findDocument (Document document)
    {
        DocumentRow find (TreeBox.Folder folder)
        {
            DocumentRow result;
            
            for (int c; c < folder.rowCount (); c ++)
            {
                Row row = folder.row (c);
                
                if ((TreeBox.Folder) row)
                {
                    if ((result = find ((TreeBox.Folder) row)) !== null)
                        return result;
                }
                else if ((DocumentRow) row && ((DocumentRow) row).base === document)
                    return (DocumentRow) row;
            }
            
            return null;
        }
        
        return find (root);
    }
    
    Folder findDocumentFolder (Document document)
    {
        DocumentRow row = findDocument (document);
        
        if (!row)
            return null;
        return (Folder) row.parent ();
    }
}

/+
class Tool
{
    bit open = true;
    
    /** The name of this tool. */
    abstract char[] name ();
    
    void draw (ProjectView view, int x, inout int y)
    {
        view.drawFolderHead (x, y, name (), open, false, false);
    }
    
    Object findHit (inout int y)
    {
        if (y == 0)
            return this;
        y --;
        return null;
    }

    /** Show menu for this tool. */
    void showMenu (ProjectView view)
    {
    }
    
    /** Get the scrolling extents in pixels along x and units along y. */
    void extents (ProjectView view, inout int x, inout int y)
    {
        y ++;
        x = imax (x, view.baseX + view.textWidth (name ()));
    }
}

class FilesTool : Tool
{
    override char[] name ()
    {
        return "Files";
    }
    
    override Object findHit (inout int y)
    {
        Object result;
        
        if ((result = super.findHit (y)) !== null)
            return result;
        if (!open)
            return null;
        return findHitFolder (global.root, y);
    }
    
    Object findHitFolder (Folder folder, inout int y)
    {
        for (int c; c < folder.folders.length; c ++)
        {
            Folder f = folder.folders [c];

            if (y == 0)
                return f;
            y --;
            if (f.open)
            {
                Object o = findHitFolder (f, y);

                if (o !== null)
                    return o;
            }
        }

        if (y < folder.documents.length)
            return folder.documents [y];
        y -= folder.documents.length;

        return null;
    }
    
    override void draw (ProjectView view, int x, inout int y)
    {
        super.draw (view, x, y);
        if (!open)
            return;
        drawFolder (view, global.root, x + view.folderCrossWidth (), y, true);
    }
    
    void drawFolder (ProjectView view, Folder folder, int x, inout int y, bit issub)
    {
        with (view)
        {
            Color dotColor = global.projectFolderBackground.blend (AColor (0, 0, 0), 128);
            int m = global.projectFontHeight;
            int bx = view.baseX + x - 6;
    
            for (int c; c < folder.folders.length; c ++)
            {
                Folder f = folder.folders [c];
                int sx = view.baseX + x;
                int sy = view.baseY + y * m;
                int mx = sx + 4;
                int my = sy + m / 2 - 1;
    
                drawFolderHead (x, y, f.name, f.open, issub, c == folder.folders.length - 1 && folder.documents.length == 0);
    
                if (f.open)
                    drawFolder (view, f, x + folderCrossWidth (), y, true);
                    
                if (c != folder.folders.length - 1 || folder.documents.length)
                {
                    penStyle ("...");
                    penWidth (1);
                    penColor (dotColor);
                    line (bx, sy + m, bx, baseY + y * m);
                }
            }
    
            brushClear ();
            for (int c; c < folder.documents.length; c ++)
            {
                Document d = folder.documents [c];
                char [] t = d.name ();
                int sx = baseX + x;
                int sy = baseY + y * m;
                int mx = sx + 4;
                int my = sy + m / 2 - 1;
    
                if (issub)
                {
                    penStyle ("...");
                    penWidth (1);
                    penColor (dotColor);
                    line (bx, sy, bx, (c < folder.documents.length - 1 ? sy + m : my));
                    line (bx, my, sx, my);
                }
    
                if (d === global.view.document)
                {
                    penColor (Color.Black);//global.projectSelection);
                    penWidth (1);
                    penStyle ("---");
                    brushColor (global.projectSelection);
                    rect (imax (baseX + x - 3, 0), y * m + baseY - 1, visualWidth (), (y + 1) * m + baseY);
                    brushClear ();
                }
    
                if (d.modified)
                    textColor (global.projectFileModified);
                else
                    textColor (global.projectFile);
                textPrint (baseX + x, baseY + y * m, t);
    
                y += 1;
            }
        }
    }
    
    override void showMenu (ProjectView view)
    {
        MenuCommand menu;
        
        with (menu = new MenuCommand ())
        {
            add ("New File", "FileNew");
            add ("Open File", "FileOpen");
            menu.separator ();
            menu.add ("New Project", "ProjectNew");
            menu.add ("Open Project", "ProjectOpen");
            menu.add ("Save Project", "ProjectSave");
            menu.add ("Statistics", "ProjectStats");
            separator ();
            add ("New Folder", "ProjectNewFolder");
        }
        
        global.view.currentFolder (global.root);
        menu.run (global.view);
        global.view.currentFolder (null);
    }
    
    override void extents (ProjectView view, inout int x, inout int y)
    {
        super.extents (view, x, y);
        if (open)
            folderExtents (view, global.root, x, y);
    }
    
    void folderExtents (ProjectView view, Folder folder, inout int x, inout int y)
    {
        for (int c; c < folder.folders.length; c ++)
        {
            Folder f = folder.folders [c];

            x = imax (x, view.baseX + view.textWidth (f.name));
            if (f.open)
            {
                view.baseX += view.folderCrossWidth ();
                folderExtents (view, f, x, y);
                view.baseX -= view.folderCrossWidth ();
            }
        }

        for (int c; c < folder.documents.length; c ++)
            x = imax (x, view.baseX + view.textWidth (folder.documents [c].name ()));

        y += folder.folders.length + folder.documents.length;
    }
}

class ProjectView : Canvas
{
    int offset = 0;
    int baseX = 2;
    int baseY = 2;
    int offsetx = 0;
    MenuCommand menu, folderMenu;
    bit deferPaint = false;
    Tool[] tools;

    this (Control parent)
    {
        super (parent);
        onPaint.add (&doPaint);
        onLButtonDown.add (&doLButtonDown);
        onRButtonUp.add (&doRButtonUp);
        onVScroll.add (&doVScroll);
        onHScroll.add (&doHScroll);
        suggestWidth (150);
        
        tools ~= new FilesTool;

        menu = new MenuCommand ();
        menu.add ("New", "FileNew");
        menu.add ("Open", "FileOpen");
        menu.add ("Close", "FileClose");
        menu.add ("Save", "FileSave");
        menu.add ("Rename", "FileRename");
        menu.separator ();
        menu.add ("New Project", "ProjectNew");
        menu.add ("Open Project", "ProjectOpen");
        menu.add ("Save Project", "ProjectSave");
        menu.add ("Statistics", "ProjectStats");
        menu.separator ();
        menu.add ("New Folder", "ProjectNewFolder");

        with (folderMenu = new MenuCommand ())
        {
            add ("New File", "FileNew");
            add ("Open File", "FileOpen");
            menu.separator ();
            menu.add ("New Project", "ProjectNew");
            menu.add ("Open Project", "ProjectOpen");
            menu.add ("Save Project", "ProjectSave");
            menu.add ("Statistics", "ProjectStats");
            separator ();
            add ("New Folder", "ProjectNewFolder");
            add ("Rename Folder", "ProjectRenameFolder");
            add ("Delete Folder", "ProjectDeleteFolder");
        }
    }

    override void paint ()
    {
        if (deferPaint)
            return;
        super.paint ();
    }

    Object findHit (int y)
    {
        Object result;
        
        y = y / global.projectFontHeight + offset;
        for (int c; c < tools.length; c ++)
            if ((result = tools [c].findHit (y)) !== null)
                return result;
        return null;
    }

    void doLButtonDown (Event e)
    {
        Object obj = findHit (e.y);

        if (obj === null)
            return;
        else if (cast (Document) obj !== null)
            global.view.switchDocument (cast (Document) obj);
        else if (cast (Folder) obj !== null)
        {
            (cast (Folder) obj).open = !(cast (Folder) obj).open;
            paint ();
        }
        else if (cast (Tool) obj !== null)
        {
            (cast (Tool) obj).open = !(cast (Tool) obj).open;
            paint ();
        }
    }

    void doRButtonUp (Event e)
    {
        Object obj = findHit (e.y);

        if (obj === null)
            return;
        else if (cast (Document) obj !== null)
        {
            int orderedCount = global.view.ordered.length;

            global.view.switchDocument (cast (Document) obj);
            menu.run (global.view);
        }
        else if (cast (Folder) obj !== null)
        {
            int orderedCount = global.view.ordered.length;

            global.view.currentFolder (cast (Folder) obj);
            folderMenu.run (global.view);
            for (int c = orderedCount; c < global.view.ordered.length; c ++)
                (cast (Folder) obj).addDocument (global.view.ordered [c]);
            global.view.currentFolder (null);

            global.projectView.paint ();
        }
        else if (cast (Tool) obj !== null)
            (cast (Tool) obj).showMenu (this);
    }
    
    void extents (out int x, out int y)
    {
        for (int c; c < tools.length; c ++)
            tools [c].extents (this, x, y);
    }
    
    void drawFolderHead (int x, inout int y, char[] name, bit open, bit issub, bit islast)
    {
        Color dotColor = global.projectFolderBackground.blend (AColor (0, 0, 0), 128);
        int m = global.projectFontHeight;
        int sx = x + baseX;
        int sy = y * m + baseY;
        int bx = baseX + x - 6;
        int mx = sx + 4;
        int my = sy + m / 2 - 1;

        penClear ();
        brushColor (global.projectFolderBackground);
        rect (sx, sy, width (), baseY + (y + 1) * m);
        brushClear ();

        /* Draw the parent cross. */
        if (issub)
        {
            penStyle ("...");
            penWidth (1);
            penColor (dotColor);
            line (bx, sy, bx, islast ? my : sy + m);
            line (bx, my, sx, my);
        }

        /* Draw the text label. */
        textColor (global.projectFolder);
        textPrint (mx + 8, sy, name);

        /* Draw the cross. */
        penStyle ("---");
        penWidth (1);
        penColor (AColor (0, 0, 0));
        rect (mx - 4, my - 4, mx + 5, my + 5);
        line (mx - 2, my, mx + 3, my);
        if (!open)
            line (mx, my - 2, mx, my + 3);
         
        y ++;
    }

    /** Return the width of a folder cross. */
    int folderCrossWidth ()
    {
        return 10;
    }
  
    /** Draw the "Files" folder, which opens to all project files. */
    void drawFilesFolder (int x, int y)
    {
        for (int c; c < tools.length; c ++)
            tools [c].draw (this, x, y);
    }

    void doPaint (Event e)
    {
        Document [] list = global.view.documents;
        int pageHeight = height () / global.projectFontHeight;

        beginPaint ();
        clear (global.projectBackground);
        textFont (global.projectFont);

        int maxx, maxy;
        
        extents (maxx, maxy);
        maxx += 20;

        offset = imin (offset, maxy - pageHeight + 1);
        offset = imax (0, offset);
        vscrollRange (0, maxy);
        vscrollPage (pageHeight);
        vscrollPoint (offset);

        offsetx = imax (0, imin (offsetx, maxx - width ()));
        hscrollRange (0, maxx);
        hscrollPage (width ());
        hscrollPoint (offsetx);

        int y = -offset;

        for (int c; c < tools.length; c ++)
            tools [c].draw (this, -offsetx, y);

        endPaint ();
    }

    void doVScroll (Event e)
    {
        offset = e.scrollDest;
        paint ();
    }

    void doHScroll (Event e)
    {
        offsetx = e.scrollDest;
        paint ();
    }
}
+/
